
rm(list=ls())
library(dplyr)
library(Synth)

dat = read.csv("../../RAS.csv", header=TRUE)

dat$country = as.character(dat$country)

dat = dat[, c("country", "ctyid", "year", "polity", "gdppc",  "rdi", "muslim", 
              "state", "social")]

set.seed(12345)

# combine MENA countries with Uprisings
list.mena = c("Tunisia","Egypt","Libya","Yemen","Bahrain","Syria","Algeria","Iraq","Jordan","Kuwait","Morocco","Oman","Sudan",
              "Mauritania","Saudi Arabia","Djibouti","Western Sahara","Palestinian Authority (West Bank)")
list.major = c("Bahrain","Syria","Algeria","Iraq","Jordan","Kuwait","Morocco","Oman","Sudan")
list.regime.change = c("Tunisia","Egypt","Libya","Yemen")

mena = aggregate(dat[dat$country %in% list.mena , c("polity","gdppc","rdi","state","social")],
                 by=list(year=dat[dat$country %in% list.mena , "year"]),
                 FUN = mean, na.rm=TRUE)
major = aggregate(dat[dat$country %in% list.major , c("polity","gdppc","rdi","state","social")],
                  by=list(year=dat[dat$country %in% list.major , "year"]),
                  FUN = mean, na.rm=TRUE)
regime.change = aggregate(dat[dat$country %in% list.regime.change , c("polity","gdppc","rdi","state","social")],
                          by=list(year=dat[dat$country %in% list.regime.change , "year"]),
                          FUN = mean, na.rm=TRUE)


mena = data.frame(country="Uprising Countries", ctyid=1000, muslim=1, mena)
major = data.frame(country="Major Protest Countries", ctyid=1000, muslim=1, major)
regime.change = data.frame(country="Regime Change Countries", ctyid=1000, muslim=1, regime.change)


### CHANGE HERE TO SPECIFY WHICH COUNTRIES TO COMBINE
#dat = rbind(dat, mena)
#add.name = " (Uprising Countries)"
#treatment.identifier = "Uprising Countries"

dat = rbind(dat, major)
add.name = " (Major Protest)"
treatment.identifier = "Major Protest Countries"

#dat = rbind(dat, regime.change)
#add.name = " (Regime Change)"
#treatment.identifier = "Regime Change Countries"


predictors.global = c("polity","gdppc","rdi", "state", "social")


##############################################################
# Create not.in function to make life easier
##############################################################
`%not.in%` <- function(x,y)!('%in%'(x,y))


##############################################################
# Delete Incomplete Countries as Preparation for Synthetic Control
##############################################################

num.missing <- function(x) {
  return (sum(is.na(x)))
}

dat.temp = dat
out = aggregate(x=dat.temp, by=list(country=dat.temp$country), FUN=num.missing)

has.missing = rowSums((out[,-1])) > 0
countries.without.missing = out$country[!has.missing]

dat = dat[dat$country %in% countries.without.missing,]


##############################################################
# synthetic control estimation
##############################################################

build.data <- function(dat.synth, predictors, dependent, treatment.identifier, controls.identifier, 
                       time, time.pretreat) {
  
  return(
    dataprep(
      foo = dat.synth,
      predictors = predictors,
      predictors.op = "mean",
      dependent = dependent,
      unit.variable = "ctyid",
      time.variable = "year",
      treatment.identifier = treatment.identifier,
      controls.identifier = controls.identifier,
      time.predictors.prior = time.pretreat,
      time.optimize.ssr = time.pretreat,
      unit.names.variable = "country",
      time.plot = time
    )
  )
}


# Draw Gap
draw.gap <- function(diff.mat, x.range, ylim, ylab, time.v, text.x, text.y, text) {
  plot(x.range, diff.mat[nrow(diff.mat),],
       type="l", ylim=ylim, 
       col=rgb(0,0,0,alpha=1), lty="solid",
       ylab = ylab,
       xlab ="Year",
       lwd=7,
       axes = FALSE,
       cex.lab = 1.5
  )
  axis(side=1, at=seq(time[1], time[length(time)], 2))
  axis(side=2, at=seq(ylim[1], ylim[2], 5))
  
  abline(v=time.v-1, lty="dotted", lwd=1.5)
  abline(v=time.v, lty="dotted", lwd=1.5)
  text(text.x-.5, text.y, text, cex=1.4)
  
  abline(h=0, lty="dotted")
  
  for (i in 1:(nrow(diff.mat)-1)) {
    lines(x.range, diff.mat[i,], lwd=.1)  
  }
}


##############################################################
# defining global parameters
##############################################################


time = 1990:2014
time.pretreat = 1990:2010

# global parameters for graphic
width = 3200
height = 2400
res = 300

list.countries = unique(dat$country[dat$country %not.in% c(treatment.identifier, list.regime.change, list.major)])
list.countries = c(list.countries, treatment.identifier)


##############################################################
# social
##############################################################

dependent = "social"
predictors = unique(c(predictors.global, dependent))

excluded = c("Brazil", "France", "Gabon", "India", "Ireland", "Mongolia", "Nepal", "Portugal", "UAE", "USA", "Vietnam")

list.treatment.identifier = list.countries[list.countries %not.in% excluded]

#so doesn't have to start from beginning
#list.treatment.identifier = list.countries[-(1:which(list.countries==excluded[length(excluded)]))]


diff.mat = matrix(NA, nrow=length(list.treatment.identifier), ncol=length(time))

for (i in 1:length(list.treatment.identifier)) {
  treatment.identifier = list.treatment.identifier[i]
  
  controls.identifier = unique(dat$country[dat$country %not.in% c(treatment.identifier, list.regime.change, list.major)])  
  
  dataprep.out <- build.data(dat.synth = dat,
                             predictors = predictors, 
                             dependent = dependent, 
                             treatment.identifier, 
                             controls.identifier,
                             time,
                             time.pretreat)
  
  print(paste0(i," = ",treatment.identifier))
    
  synth.out <- synth(dataprep.out)
  Y0 = (dataprep.out$Y0%*%synth.out$solution.w)
  Y1 = dataprep.out$Y1plot
  diff.mat[i,] = Y1-Y0
}


row.names(diff.mat) <- list.treatment.identifier
write.csv(diff.mat, file=paste0("In-Space Societal", add.name, ".csv"))

diff.mat = read.csv(paste0("In-Space Societal", add.name, ".csv"), row.names=1, header=TRUE)

#delete extreme countries with no reasonable synthetic control
diff.mat = diff.mat[abs(diff.mat[,1]) < 5,]

png(filename = paste0("Plot In-Space Societal", add.name, ".png"), width=width, height=height, res=res)
draw.gap(diff.mat, 
         x.range = time, 
         ylim = c(-30,30), 
         ylab = "Gap in Societal Discrimination of Religious Minorities",
         time.v = 2011, 
         text.x = 2011, 
         text.y = -8, 
         text = "Arab\nUprisings")
dev.off()


#p-value = 0.2244898
sum(abs(diff.mat[,25]) > abs(diff.mat[nrow(diff.mat),25]))/nrow(diff.mat)
